
package easik.sketch.util.Export;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.ListIterator;

import easik.sketch.util.Export.Components.ExportAttribute;
import easik.sketch.util.Export.Components.ExportEdge;
import easik.sketch.util.Export.Components.ExportTable;
import easik.sketch.util.Export.Components.ExportUniqueKey;
import easik.sketch.util.Export.Constraints.ExportConstraint;
import easik.sketch.util.Export.Constraints.ExportTriggerFormatter;


/**
 * Sketch to Schema is a class which will accept a sketch and then
 * generate an SQL schema. It can currently create a schema without constraints.  
 * Constraints will be implemented later using SQL triggers.  
 * 
 * @author Rob Fletcher 2005
 * @author Vera Ranieri 2006
 * @version 2006-06-16 Vera Ranieri
 */
public class SketchToSchema {

	/**
	 * A list of all the table to be defined
	 */
	private LinkedList<String> _finalTables;
	/**
	 * A list of al the procedures to be defined.
	 */
	private ArrayList<String> _procedures;
	
	/**
	 * The string <b>CREATE DATABASE</b><it> name</it>
	 */
	private String _createString;
	/**
	 * The string <b>USE <b><it>name</it>
	 */
	private String _useString;
	
	/**
	 * The Trigger formatter associated with this sketch
	 */
	private ExportTriggerFormatter _triggers;
	
	/**
	 * The export handler associated with this sketch
	 */
	private  ExportHandler _eHandler;
	
	/**
	 * This is the constructor that takes the ExportHandler containing information via 
	 * the XML file (that is, bypassing all sketch information) and provides the output schema.
	 * 
	 * @param sh The ExportHandler containg information about the sketch gleaned from the XML file.
	 * @param name The name of the database to be created.
	 */
	public SketchToSchema( ExportHandler sh, String name ) {	
		
		_eHandler = sh;
		
		for(ExportConstraint exCont : _eHandler.getConstraints()){
			exCont.set_db_name(name);
		}
		
		setCreateString(name);
		setUseString(name);

		// Our output and work structures
		_finalTables = new LinkedList<String>();
		_procedures = new ArrayList<String>();
		_triggers = new ExportTriggerFormatter();
		
		setTables();
		setConstraints();
		_triggers.setTriggers();
		
		//TODO: method needs to be removed once they are fully enforced.
		printSchemaInfo();
		
	}
	
	/**
	 * A method to set all tables associated with the schema.  Setting the tables includes setting all primary, foreign, and 
	 * unique keys for the table.
	 *
	 */
	private void setTables(){
		
		//setAllUnMarked(eHandler.getTables());
		
		//TODO: what if there are no sinks?? error checking.
		LinkedList<ExportTable> workingSet = getSinks( _eHandler.getTables() );	
		
					
		// Loop until our working set has been processed
		while (workingSet.size() != 0) {	
			ListIterator<ExportTable> listIterator = workingSet.listIterator();
														
			// For each element
			while (listIterator.hasNext()) {				
				
				ExportTable currentTable= (ExportTable)listIterator.next();
					// Are the current node's neighbors all marked? 
					if ( areNeighborsMarked( _eHandler.getTables(), currentTable ) ) {
						
						// If they are, we can add them as a table to our shema
						_finalTables.add( newTable( currentTable ) );					
						
						// Remove this element from the working set
						currentTable.setMarked(true);
						listIterator.remove();
					
						// Add its parents (which are not marked) to the set at the iterator					
						addUnmarkedParents( _eHandler.getTables(), workingSet, listIterator, currentTable );
					}	
				
			}					
		}
	}
	
	/**
	 *	Find all the tables in the sketch that do not have any outgoing
	 *	edges.
	 *
	 *	@param tables The hashmap of all tables contained in this sketch
	 *	@return A linked list of the current sinks
	 */
	private LinkedList<ExportTable> getSinks(HashMap tables) {		
		
		Iterator setIterator =tables.keySet().iterator();
		LinkedList<ExportTable> toReturn = new LinkedList<ExportTable>();
	
		ExportTable currentTable;
		while( setIterator.hasNext() ){
			currentTable = (ExportTable)tables.get(setIterator.next());
		
			if (currentTable.getEdges().size()==0) {
				toReturn.add( currentTable );
			}
		}
		
		return toReturn;
	}
		
	/**
	 * A helper method which will check to see if the neighbours of a current table have been
	 * marked. 
	 * 
	 * @param tables The tables of this sketch
	 * @param t The current table for which it must be determined whether neighbours are marked.
	 * @return The truth about the marked neighbors (true if they are marked)
	 */
	private boolean areNeighborsMarked(HashMap tables,ExportTable t) {
		
		ArrayList edges = t.getEdges();
		int size = edges.size();
		
		
		ExportEdge tempEdge;
		
		for(int i = 0; i< size; i++){			
			
			tempEdge = (ExportEdge)edges.get(i);
			if(!((ExportTable)tables.get(tempEdge.getTarget())).isMarked())
							return false;	
		}
		return true;
	}
	

	/**
	 * Find all the unmarked parents of marked nodes, add them to the provided
	 * listIterator.
	 *  
	 * @param tables The set of tables for the sketch
	 * @param workingSet The tables which are currently 'in play'
	 * @param listIterator The list iterator containing where we are in the nodes
	 * @param currentTable The table whose unmarked parents we're looking for
	 */
	 private void addUnmarkedParents(	HashMap tables, LinkedList workingSet,
	 															ListIterator<ExportTable> listIterator, 
	 															ExportTable currentTable) {
		 ArrayList edges;
		 int size;
		
		 Iterator it = tables.keySet().iterator();
		 
		 ExportTable tempTable;
		 ExportEdge tempEdge;
		 
		 while(it.hasNext()){
			 tempTable = (ExportTable)tables.get(it.next());
			 if(!tempTable.isMarked()){
				 edges = tempTable.getEdges();
				 size = edges.size();
				 
				 for(int i =0; i< size; i++){					
						
						tempEdge = (ExportEdge)edges.get(i);
						if (tempEdge.getTarget().equals(currentTable.getName()) && !workingSet.contains( tempTable )) {								
							listIterator.add( tempTable );
						}
					
				}
				 
			 }
		 }
		
		
		
		
	 }
	
	/**
	 * Given a table, create a SQL table (which is a string).
	 *
	 * @param table The table which is getting its own SQL table
	 * @return A String containing a light description of the table (keys and foreign keys)
	 * @since 2006-05-17, Vera Ranieri
	 */
	private String newTable(ExportTable table) {
		
		// Create the table				
		String newTable = ExportConstants.CREATE_TABLE + table.getName() + " (" + ExportAttribute.getAttributeString(table)
							+ ExportEdge.getForeignKeyString(table)
							+ ExportUniqueKey.getUniqueKeyString(table) 
							+ ExportTable.getPrimaryKeyString(table) +") ";	
		
		if(_eHandler.getPlatform().equals(ExportConstants.MY_SQL))
			newTable += ExportConstants.MySQL_ENGINE;
		
		return newTable;		
	}		
	
	/**
	 * Getter method for the constructed set of tables
	 * 
	 * @return The tables needed in the schema
	 * @since 2006-05-10, Vera Ranieri
	 *
	 */
	public LinkedList<String> getTables(){
		return _finalTables;
	}
	
	/**
	 * Getter method for the set of constraints of the schema.
	 * 
	 * @return The constraints needed in the schema
	 * @since 2006-05-10, Vera Ranieri
	 *
	 */
	public ArrayList<String> getConstraints(){
		return _procedures;
	}
	
	/**
	 * Getter method for all triggers formed from the table.  Triggers are amalgamated into one arraylist for easy
	 * access for passing to the jdbc connector.
	 * 
	 * @return The arraylist of all triggers
	 * @since 2006-06-16, Vera Ranieri
	 */
	public ArrayList<String> getTriggers(){
		return _triggers.getAllTriggers();
	}
	
	/**
	 * Getter method for the <b>CREATE DATABASE <it>name</it></b> SQL string
	 * 
	 * @return The "CREATE DATABASE..." string
	 *
	 * @since 2006-05-10, Vera Ranieri
	 */
	public String getCreateString() {
		return _createString;
	}

	/**
	 * Setter method for the <b>CREATE DATABASE <it>name</it> </b>SQL string.  Takes the name of the database as the 
	 * input, and appends it to the appropriate SQL string that will create a database.
	 * 
	 * @param name  The name of the database
	 * @since 2006-05-17, Vera Ranieri
	 */
	public void setCreateString(String name) {
		_createString = ExportConstants.CREATE_DB + name +"; ";
	}

	/**
	 * Getter method for the <b>USE <it>database</it></b>  SQL string. 
	 *  
	 * @return The "USE ... " string
	 * @since 2006-05-17, Vera Ranieri
	 */
	public String getUseString() {
		return _useString;
	}

	/**
	 * Sets the <b>USE <it>database</it></b> string for the schema.
	 * 
	 * @param name The name of the database
	 * @since 2006-05-17, Vera Ranieri
	 */
	public void setUseString(String name) {
		_useString = ExportConstants.USE + name +"; ";
	}
	
	

	
	/**
	 * Sets the constraint string for this schema
	 *
	 * @since 2006-06-09, Vera Ranieri
	 */
	private void setConstraints(){
		
		ArrayList<ExportConstraint> eConstraints = _eHandler.getConstraints();
		
		for(ExportConstraint con: eConstraints){
			con.setConstraintStrings();
			
			_procedures.addAll( con.getConstraintStrings());
			_triggers.addTrigger(con);
		}
		
	}
	
	
	
	/**
	 * Temporary method to print schema info.  Will be removed once all testing is completed
	 * 
	 * 
	 * @since 2006-05-18, Vera Ranieri
	 * 
	 *
	 */
	private void printSchemaInfo(){
		System.out.println();
		System.out.println("\n Our Schema:");
		Iterator i = _finalTables.iterator();
		while(i.hasNext()){
			
			String t = (String) i.next();
			System.out.println("Table: " + t);
			System.out.println();
		}
		
		for(int j=0; j<_procedures.size(); j++){
			String c = (String)_procedures.get(j);
			System.out.println("Constraint: " + c);

			System.out.println();
		}
		
		ArrayList trigs = _triggers.getAllTriggers();
		for(int k = 0; k<trigs.size(); k++){
			String c = (String)trigs.get(k);
			System.out.println("Trigger: " + c);
		}
		
		
	}
	
}